﻿using System;
using System.Collections.Generic;
using System.IO.Ports;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using System.Windows.Threading;
using AZVIC.Ei8htPOS.POSRetail.Constants;
using AZVIC.Ei8htPOS.Entities;
using AZVIC.Ei8htPOS.Utilities;
using System.IO;

namespace AZVIC.Ei8htPOS.POSRetail.OCBC
{
    /// <summary>
    /// Interaction logic for OCBCTerminal.xaml
    /// </summary>
    public partial class OCBCTerminal : Window
    {
        private SerialPort comPort = new SerialPort();
        private OCBCHelper.OCBCTransactionRequest request;
        private string ResponseMessage = string.Empty, RequestMessage = string.Empty, LogMessage = string.Empty;
        private int Timeout { get; set; }
        private int Retries { get; set; }
        private bool IsAckReceived = false;
        private bool ACKResult;
        private decimal Amount = 0;

        public OCBCTerminal()
        {
            InitializeComponent();
        }

        public OCBCTerminal(decimal amount)
        {
            InitializeComponent();
            Amount = amount;
            SendData();
        }

        private void btnRetry_Click(object sender, RoutedEventArgs e)
        {
            SendData();
        }

        private void btnCancel_Click(object sender, RoutedEventArgs e)
        {
            ClosePort();
            this.Close();
        }

        private void SendData()
        {
            txtDisplayWindow.Text = string.Empty;
            RequestMessage = ResponseMessage = string.Empty;
            Retries = 2;
            IsAckReceived = false;
            ACKResult = false;
            btnRetry.IsEnabled = false;

            //Establish a connection to the serial port (COM1).
            OpenPort();

            DisplayData(OCBCHelper.MessageType.Normal, string.Format("-------------{0}------------------{1}", "Started at " + DateTime.Now.ToString(), Environment.NewLine));

            request = new OCBCHelper.OCBCTransactionRequest()
            {
                TransactionType = OCBCHelper.TransactionType.SaleFullPayment,
                ReceiptNumber = AppSession.CurrentOrders.OrderNumber,
                Amount = Amount,
                TerminalRefNo = 0,
                Currency = "SGD"
            };

            RequestMessage = request.ToString();
            WriteData(RequestMessage);
            DisplayData(OCBCHelper.MessageType.Normal, "Sent Request at " + DateTime.Now + Environment.NewLine + RequestMessage + Environment.NewLine);
        }

        #region Open & Close Port
        public bool OpenPort()
        {
            try
            {
                //first check if the port is already open
                //if its open then close it
                if (this.comPort.IsOpen == true) comPort.Close();

                //set the properties of our SerialPort Object
                comPort.BaudRate = int.Parse(AppSession.OCBCTerminalBaudRate);
                comPort.DataBits = int.Parse(AppSession.OCBCTerminalDataBits);
                comPort.StopBits = (StopBits)Enum.Parse(typeof(StopBits), AppSession.OCBCTerminalStopBits);
                comPort.Parity = (Parity)Enum.Parse(typeof(Parity), AppSession.OCBCTerminalParity);
                comPort.PortName = AppSession.OCBCTerminalPortName;
                comPort.Handshake = Handshake.None;
                comPort.ReadTimeout = 2000;
                comPort.WriteTimeout = 2000;
                //now open the port
                comPort.Open();
                comPort.DataReceived += new System.IO.Ports.SerialDataReceivedEventHandler(comPort_DataReceived);
                //display message
                DisplayData(OCBCHelper.MessageType.Normal, "Port opened at " + DateTime.Now + Environment.NewLine);
                return true;
            }
            catch (Exception ex)
            {
                DisplayData(OCBCHelper.MessageType.Error, ex.Message);
                return false;
            }
        }

        public bool ClosePort()
        {
            try
            {
                //first check if the port is already open
                //if its open then close it
                if (this.comPort.IsOpen == true)
                {
                    comPort.DataReceived -= comPort_DataReceived;
                    comPort.Close();
                }

                //display message
                DisplayData(OCBCHelper.MessageType.Normal, "Port Closed at " + DateTime.Now + Environment.NewLine);
                DisplayData(OCBCHelper.MessageType.Normal, string.Format("-------------{0}------------------{1}", "Ended at " + DateTime.Now.ToString(), Environment.NewLine));

                SaveTerminalLogToFile();

                return true;
            }
            catch (Exception ex)
            {
                DisplayData(OCBCHelper.MessageType.Error, ex.Message);
                return false;
            }
        }

        public void FlushBuffer()
        {
            comPort.DiscardInBuffer();
            comPort.DiscardOutBuffer();
            ClosePort();
        }
        #endregion

        #region WriteData
        public void WriteData(string msg)
        {
            //convert the message to byte array
            byte[] newMsg = HexToByte(msg);
            if (!comPort.IsOpen)
                OpenPort();
            //send the message to the port
            if(comPort.IsOpen)
            comPort.Write(newMsg, 0, newMsg.Length);
            else
                DisplayData(OCBCHelper.MessageType.Error, "Error in sending message to the Terminal. Try retry again or use Manual transaction."); 
        }

        private OCBCHelper.TwoStepProtocol WaitForResponse(string acknowledgement)
        {
            OCBCHelper.TwoStepProtocol result = OCBCHelper.TwoStepProtocol.Timeout;

            if (acknowledgement.Trim().Equals(StringValueAttribute.GetStringValue(OCBCHelper.TwoStepProtocol.Acknowledge)))
            {
                DisplayData(OCBCHelper.MessageType.Incoming, "Tml Reply ACK at " + DateTime.Now + Environment.NewLine);
                result = OCBCHelper.TwoStepProtocol.Acknowledge;
                IsAckReceived = true;
            }
            else
            {
                if (acknowledgement.Trim().Equals(StringValueAttribute.GetStringValue(OCBCHelper.TwoStepProtocol.NegativeAcknowledge)))
                {
                    DisplayData(OCBCHelper.MessageType.Incoming, "Tml Reply NACK at " + DateTime.Now + Environment.NewLine);
                    result = OCBCHelper.TwoStepProtocol.NegativeAcknowledge;
                    IsAckReceived = true;
                }
            }

            return result;
        }

        private void SendAcknowledgement()
        {
            this.WriteData(StringValueAttribute.GetStringValue(OCBCHelper.TwoStepProtocol.Acknowledge));
            ClosePort();
        }

        private void SendNegativeAcknowledgement()
        {
            ResponseMessage = string.Empty; //Reset to receive the Response message from port again
            this.WriteData(StringValueAttribute.GetStringValue(OCBCHelper.TwoStepProtocol.NegativeAcknowledge));
        }
        #endregion

        #region comPort_DataReceived
        private void comPort_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {
            ////retrieve number of bytes in the buffer
            int bytes = comPort.BytesToRead;
            //create a byte array to hold the awaiting data
            byte[] comBuffer = new byte[bytes];
            //read the data and store it
            comPort.Read(comBuffer, 0, bytes);
            //display the data to the user

            string responseString = ByteToHex(comBuffer);

            if (!IsAckReceived)
            {
                ACKResult = (WaitForResponse(responseString) == OCBCHelper.TwoStepProtocol.Acknowledge);
                while (!ACKResult && (Retries > 0))
                {
                    Retries--;
                    WriteData(request.ToString());
                    ACKResult = (WaitForResponse(responseString) == OCBCHelper.TwoStepProtocol.Acknowledge);
                }
            }
            else
            {
                ResponseMessage += responseString;
                if (ResponseMessage.Trim().Contains(StringValueAttribute.GetStringValue(OCBCHelper.TwoStepProtocol.EndOfText)))
                {
                    ProcessResponseToTransactionResult();
                }
            }
        }
        #endregion

        #region ProcessResponseToTransactionResult
        private void DisplayData(OCBCHelper.MessageType type, string msg)
        {
            LogMessage += msg;
            txtDisplayWindow.Dispatcher.BeginInvoke((Action)(() => txtDisplayWindow.Text += msg));
            txtDisplayWindow.Dispatcher.BeginInvoke((Action)(() => txtDisplayWindow.CaretIndex = txtDisplayWindow.Text.Length));
            txtDisplayWindow.Dispatcher.BeginInvoke((Action)(() => txtDisplayWindow.ScrollToEnd()));
        }

        private void SaveTerminalLogToFile()
        {
            using (StreamWriter w = File.AppendText(System.IO.Path.GetTempPath() + "OCBClog_" + DateTime.Now.ToString("yyyyMMdd") + ".txt"))
            {
                w.WriteLine(LogMessage);
            }
        }

        private void ProcessResponseToTransactionResult()
        {
            if (!string.IsNullOrEmpty(ResponseMessage))
            {
                DisplayData(OCBCHelper.MessageType.Incoming, "Received Response at " + DateTime.Now + Environment.NewLine + ResponseMessage + Environment.NewLine);

                OCBCHelper.OCBCTransactionResponse response = new OCBCHelper.OCBCTransactionResponse()
                {
                    UniqueID = Guid.NewGuid(),
                    IsActive = true,
                    AdminUserAccountID = AppSession.CurrentAdminUser.UniqueID,
                    DeviceRequest = RequestMessage,
                    DeviceResponse = ResponseMessage
                };

                DisplayData(OCBCHelper.MessageType.Incoming, "Verifying LRC at " + DateTime.Now + Environment.NewLine);
                if (response.VerifyLRC())
                {
                    DisplayData(OCBCHelper.MessageType.Incoming, "Sending ACK at " + DateTime.Now + Environment.NewLine);
                    SendAcknowledgement();
                    response.ProcessResponseString();
                }
                else
                {
                    DisplayData(OCBCHelper.MessageType.Incoming, "Sending NACK at " + DateTime.Now + Environment.NewLine);
                    SendNegativeAcknowledgement();
                }

                if (response.ResponseCode.Equals(StringValueAttribute.GetStringValue(OCBCHelper.TransactionType.Approved)))
                {
                    Action action = () =>
                    {
                        AppSession.CurrentOCBCTransaaction = response;
                        this.Tag = true;
                        this.Close();
                    };
                    Dispatcher.BeginInvoke(DispatcherPriority.Send, action);
                }
                else
                {
                    OCBCHelper.OCBCReponseCode code = (OCBCHelper.OCBCReponseCode)Enum.Parse(typeof(OCBCHelper.OCBCReponseCode), "RC07");
                    try
                    {
                        code = (OCBCHelper.OCBCReponseCode)Enum.Parse(typeof(OCBCHelper.OCBCReponseCode), "RC" + response.ResponseCode);
                    }
                    catch { }
                    string errorMessage = StringValueAttribute.GetStringValue(code);
                    btnRetry.Dispatcher.BeginInvoke((Action)(() => btnRetry.IsEnabled = true));
                    App.ShowErrorMessageBox("Transaction Declined please try again." + Environment.NewLine + errorMessage, "OCBC Terminal Response");
                }
            }
        }
        #endregion

        #region HexToByte

        public byte[] HexToByte(string msg)
        {
            //remove any spaces from the string
            msg = msg.Replace(" ", "");
            //create a byte array the length of the
            //divided by 2 (Hex is 2 characters in length)
            byte[] comBuffer = new byte[msg.Length / 2];
            //loop through the length of the provided string
            for (int i = 0; i < msg.Length; i += 2)
                //convert each set of 2 characters to a byte
                //and add to the array
                comBuffer[i / 2] = (byte)Convert.ToByte(msg.Substring(i, 2), 16);
            //return the array
            return comBuffer;
        }
        #endregion

        #region ByteToHex

        public string ByteToHex(byte[] comByte)
        {
            //create a new StringBuilder object
            StringBuilder builder = new StringBuilder(comByte.Length * 3);
            //loop through each byte in the array
            foreach (byte data in comByte)
                //convert the byte to a string and add to the stringbuilder
                builder.Append(Convert.ToString(data, 16).PadLeft(2, '0').PadRight(3, ' '));
            //return the converted value
            return builder.ToString().ToUpper();
        }
        #endregion

        private void btnManual_Click_1(object sender, RoutedEventArgs e)
        {
            btnOK.IsEnabled = true;
            btnRetry.IsEnabled = false;
            txtRefNumber.BorderThickness = new Thickness(1);
            txtRefNumber.BorderBrush = Brushes.Red;
        }

        private void btnOK_Click_1(object sender, RoutedEventArgs e)
        {
            if (string.IsNullOrEmpty(txtRefNumber.Text))
            {
                App.ShowMessageBox("Please enter Reference Number once the transaction is complete","Ei8ht POS");
                return;
            }

            OCBCHelper.OCBCTransactionResponse response = new OCBCHelper.OCBCTransactionResponse()
            {
                UniqueID = Guid.NewGuid(),
                IsActive = true,
                AdminUserAccountID = AppSession.CurrentAdminUser.UniqueID,
                DeviceRequest = "Manual Request",
                DeviceResponse = string.Empty,
                NetAmount = Amount,
                ECRReferenceNo = txtRefNumber.Text.Trim(),
                ResponseType = OCBCHelper.TransactionType.Success,
                ResponseCode = "00"
            };
            AppSession.CurrentOCBCTransaaction = response;
            this.Tag = true;
            this.Close();

        }
    }
}
